<?php
/**
 * Nightly AI Retraining Cron Script
 *
 * This script triggers periodic retraining of the question‑generation
 * model. The heavy lifting is delegated to a Python script that lives
 * at /ai/retrain.py. The script:
 *   1. Pulls the latest training data from storage/training/raw.
 *   2. Fine‑tunes the model via Hugging Face CLI.
 *   3. Records metrics and uploads the new model to /ai/models.
 *   4. Sends an email report to administrators.
 *
 * Execute via cron:  30 2 * * * php /cron/retrain_ai.php
 */

declare(strict_types=1);

require_once __DIR__ . '/../services/TrainingUploadService.php';
require_once __DIR__ . '/../utils/Mailer.php';

$start = microtime(true);
$log   = [];

function run(string $cmd): array
{
    $output = [];
    $status = 0;
    exec($cmd . ' 2>&1', $output, $status);
    return [$status, implode("\n", $output)];
}

[$status, $msg] = run('python3 /ai/retrain.py');

$log[] = "[AI Retrain] Exit code: {$status}";
$log[] = $msg;

$duration = round(microtime(true) - $start, 2);
$log[] = "[AI Retrain] Completed in {$duration} seconds.";

Mailer::send(
    getenv('ADMIN_EMAIL'),
    'Nightly AI Retraining Report',
    implode("<br>", array_map('htmlspecialchars', $log))
);

echo "Retraining finished ({$duration}s)\n";
?>