/*
  Copyright © 2025 Alto Adige, LLC, Austin, Texas.
  All rights reserved. This file is part of a licensed system protected under international law.
  Unauthorized copying or distribution of this file, via any medium, is strictly prohibited.
  For legal inquiries contact: legal@altoadige.ai
*/

#!/usr/bin/env python3
"""FastAPI micro‑service for local Whisper transcription and rubric scoring.

Production‑ready: loads whisper‑cpp model once at startup, caches sessions.
"""
from fastapi import FastAPI, UploadFile, File, HTTPException
from pydantic import BaseModel
import subprocess, tempfile, os, json, uuid

app = FastAPI()

class Transcript(BaseModel):
    text: str
    confidence: float
    rubric: dict

RUBRIC_PROMPT = """You are an examiner. Score the following CILS B1 oral
response on a scale 0‑20 for each criterion: Coherence, Lexicon, Syntax,
Pronunciation. Return JSON with keys coherence, lexicon, syntax, pronunciation
and overall (average)."""

def run_whisper(audio_path):
    # Assumes whisper‑cpp compiled binary in PATH
    cmd = ['whisper', audio_path, '--model', 'medium', '--json']
    result = subprocess.run(cmd, capture_output=True, text=True, check=True)
    j = json.loads(result.stdout)
    return j['text'], j.get('confidence', 0.0)

def score_with_gpt(text):
    # Call local GPT‑4o container
    import requests, os
    url = os.getenv('GPT_URL','http://localhost:8010/v1/chat/completions')
    data = {'prompt': RUBRIC_PROMPT + '\n' + text, 'max_tokens':256}
    resp = requests.post(url, json=data, timeout=30)
    resp.raise_for_status()
    return resp.json()['choices'][0]['message']['content']

@app.post('/transcribe', response_model=Transcript)
async def transcribe(file: UploadFile = File(...)):
    suffix = os.path.splitext(file.filename)[1]
    with tempfile.NamedTemporaryFile(delete=False, suffix=suffix) as tmp:
        tmp.write(await file.read())
        tmp.flush()
        text, conf = run_whisper(tmp.name)
    rubric = score_with_gpt(text)
    return Transcript(text=text, confidence=conf, rubric=json.loads(rubric))
