"use strict";
/*
  Copyright 2018 Google LLC

  Use of this source code is governed by an MIT-style
  license that can be found in the LICENSE file or at
  https://opensource.org/licenses/MIT.
*/
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.askExtensionsToCache = void 0;
const assert_1 = __importDefault(require("assert"));
const inquirer_1 = __importDefault(require("inquirer"));
const glob_1 = __importDefault(require("glob"));
const ora_1 = __importDefault(require("ora"));
const upath_1 = __importDefault(require("upath"));
const errors_1 = require("../errors");
const constants_1 = require("../constants");
// The key used for the question/answer.
const name = 'globPatterns';
/**
 * @param {string} globDirectory The directory used for the root of globbing.
 * @return {Promise<Array<string>>} The unique file extensions corresponding
 * to all of the files under globDirectory.
 */
async function getAllFileExtensions(globDirectory) {
    const files = await new Promise((resolve, reject) => {
        // Use a pattern to match any file that contains a '.', since that signifies
        // the presence of a file extension.
        (0, glob_1.default)('**/*.*', {
            cwd: globDirectory,
            nodir: true,
            ignore: [
                ...constants_1.constants.ignoredDirectories.map((directory) => `**/${directory}/**`),
                ...constants_1.constants.ignoredFileExtensions.map((extension) => `**/*.${extension}`),
            ],
        }, (error, files) => {
            if (error) {
                reject(error);
            }
            else {
                resolve(files);
            }
        });
    });
    const extensions = new Set();
    for (const file of files) {
        const extension = upath_1.default.extname(file);
        if (extension) {
            // Get rid of the leading . character.
            extensions.add(extension.replace(/^\./, ''));
        }
    }
    return [...extensions];
}
/**
 * @param {string} globDirectory The directory used for the root of globbing.
 * @return {Promise<Answers>} The answers from inquirer.
 */
async function askQuestion(globDirectory) {
    // We need to get a list of extensions corresponding to files in the directory
    // to use when asking the next question. That could potentially take some
    // time, so we show a spinner and explanatory text.
    const spinner = (0, ora_1.default)({
        text: `Examining files in ${globDirectory}...`,
        stream: process.stdout,
    }).start();
    const fileExtensions = await getAllFileExtensions(globDirectory);
    spinner.stop();
    (0, assert_1.default)(fileExtensions.length > 0, errors_1.errors['no-file-extensions-found']);
    return inquirer_1.default.prompt([
        {
            name,
            message: 'Which file types would you like to precache?',
            type: 'checkbox',
            choices: fileExtensions,
            default: fileExtensions,
        },
    ]);
}
async function askExtensionsToCache(globDirectory) {
    const answers = await askQuestion(globDirectory);
    // The return value is an array of strings with the selected values
    // and there is a default, the casting is safe.
    const extensions = answers[name];
    (0, assert_1.default)(extensions.length > 0, errors_1.errors['no-file-extensions-selected']);
    // glob isn't happy with a single option inside of a {} group, so use a
    // pattern without a {} group when there's only one extension.
    const extensionsPattern = extensions.length === 1 ? extensions[0] : `{${extensions.join(',')}}`;
    return [`**/*.${extensionsPattern}`];
}
exports.askExtensionsToCache = askExtensionsToCache;
