// resources/js/router/index.js

import { createRouter, createWebHistory } from 'vue-router'
import { useAuthStore } from '@/stores/auth'

/* 1. Manually register your Home and NotFound pages */
import Home     from '@/pages/Home.vue'
import NotFound from '@/pages/NotFound.vue'

/* 2. Auto-discover every other .vue page under /pages, but skip ChangePassword.vue */
const dynamicRoutes = Object.entries(import.meta.glob('@/pages/**/*.vue')).flatMap(
  ([filePath, loader]) => {
    // derive URL path: '@/pages/Foo/Bar.vue' → '/foo/bar'
    const routePath = filePath
      .replace(/^@\/pages/, '')
      .replace(/\.vue$/, '')
      .replace(/\/index$/, '')
      .toLowerCase()

    // skip Home (already registered)
    if (routePath === '/home') return []

    // skip our explicit ChangePassword route
    if (filePath.endsWith('/pages/auth/ChangePassword.vue')) return []

    return {
      path: routePath,
      component: loader,
      meta: {
        requiresAuth: routePath.startsWith('/app') || routePath.startsWith('/user'),
        requiresAdmin: routePath.startsWith('/admin'),
      },
    }
  }
)

/* 3. Build the full routes array */
const routes = [
  { path: '/', component: Home, name: 'Home' },

  // explicit change-password route
  {
    path: '/change-password',
    name: 'ChangePassword',
    component: () => import('@/pages/auth/ChangePassword.vue'),
    // we don’t gate it behind auth: token in URL suffices
  },

  // spread in all other auto-discovered pages
  ...dynamicRoutes,

  // catch-all 404
  { path: '/:pathMatch(.*)*', component: NotFound, name: 'NotFound' },
]

/* 4. Create the router, aware of your '/cils04/' subdirectory */
const router = createRouter({
  history: createWebHistory('/cils04/'),
  routes,
  scrollBehavior: () => ({ top: 0 }),
})

/* 5. Global guard: hydrate auth once, then protect routes */
let hasHydrated = false
router.beforeEach(async (to, _from, next) => {
  const auth = useAuthStore()

  if (!hasHydrated) {
    await auth.hydrate()
    hasHydrated = true
  }

  if (to.meta.requiresAuth && !auth.isAuthenticated) {
    return next({ path: '/login', replace: true })
  }

  if (to.meta.requiresAdmin && !auth.user?.is_admin) {
    return next({ path: '/', replace: true })
  }

  next()
})

export default router
