import { defineStore } from 'pinia'

/**
 * Simple translation store.
 *  – Caches translations client-side to minimise API calls
 *  – Falls back gracefully if /api/translate is unreachable
 */
export const useI18nStore = defineStore('i18n', {
  state: () => ({
    cache: new Map(),           // { '<text>|<target>': 'translated text' }
  }),

  actions: {
    /**
     * Translate Italian text to the target language (default EN).
     * @param {string} text     – source text
     * @param {string} target   – ISO language code, e.g. 'en'
     * @returns {Promise<{text:string}>}
     */
    async translate (text, target = 'en') {
      const key = `${text}|${target}`
      if (this.cache.has(key)) {
        return { text: this.cache.get(key) }
      }

      try {
        const res  = await fetch('/api/translate', {
          method : 'POST',
          headers: { 'Content-Type': 'application/json' },
          body   : JSON.stringify({ text, target }),
        })
        const { text: translated } = await res.json()
        this.cache.set(key, translated)
        return { text: translated }
      } catch (e) {
        console.error('Translate API failed', e)
        // Graceful fallback: return original text
        return { text }
      }
    },
  },
})
