// scripts/generate-sw.js
import { generateSW } from 'workbox-build';

async function buildServiceWorker() {
  try {
    const { count, size, warnings } = await generateSW({
      // Look in your Laravel “public” folder
      globDirectory: 'public',

      // Precache the PWA manifest plus everything in public/build/*.js|css
      globPatterns: [
        'manifest.json',            // your web app manifest
        'build/**/*.{js,css}',      // Vite output
      ],

      // Exclude source‐maps and any stray workbox caches
      globIgnores: [
        '**/*.map',
        'build/.vite/**',
        'build/workbox-*.js',
      ],

      // Where to write the generated service worker
      swDest: 'public/sw.js',

      // Take control of clients as soon as the SW activates
      clientsClaim: true,
      skipWaiting: true,

      // Increase file‐size threshold if you have large assets
      maximumFileSizeToCacheInBytes: 5 * 1024 * 1024, // 5 MB

      // Runtime caching rules
      runtimeCaching: [
        {
          // HTML pages: network first, fallback to cache
          urlPattern: ({ request }) => request.destination === 'document',
          handler: 'NetworkFirst',
          options: {
            cacheName: 'html-cache',
            expiration: { maxEntries: 10, maxAgeSeconds: 86400 },
            networkTimeoutSeconds: 10,
          },
        },
        {
          // CSS files: stale-while-revalidate
          urlPattern: /\.(?:css)$/,
          handler: 'StaleWhileRevalidate',
          options: {
            cacheName: 'css-cache',
            expiration: { maxEntries: 20, maxAgeSeconds: 604800 },
          },
        },
        {
          // JS files: stale-while-revalidate
          urlPattern: /\.(?:js)$/,
          handler: 'StaleWhileRevalidate',
          options: {
            cacheName: 'js-cache',
            expiration: { maxEntries: 50, maxAgeSeconds: 604800 },
          },
        },
        {
          // Images & fonts: cache-first for offline
          urlPattern: /\.(?:png|jpg|jpeg|svg|gif|webp|woff2?|ttf|otf)$/,
          handler: 'CacheFirst',
          options: {
            cacheName: 'media-assets',
            expiration: { maxEntries: 100, maxAgeSeconds: 2592000 },
          },
        },
        {
          // API calls: network first with a timeout
          urlPattern: /\/api\/.*$/,
          handler: 'NetworkFirst',
          options: {
            cacheName: 'api-cache',
            networkTimeoutSeconds: 10,
            expiration: { maxEntries: 50, maxAgeSeconds: 300 },
          },
        },
      ],
    });

    warnings.forEach(console.warn);
    console.log(`✔ Service worker generated: precached ${count} files, totaling ${size} bytes.`);
  } catch (err) {
    console.error('❌ Service worker generation failed:', err);
    process.exit(1);
  }
}

buildServiceWorker();
