<?php
declare(strict_types=1);

namespace App\Services;

use Illuminate\Support\Facades\DB;

/**
 * Item‑Response‑Theory adaptive difficulty engine (2PL model).
 * Computes learner ability (theta) and item parameters (a, b) using
 * Bayesian updating (EAP estimator).
 *
 * No placeholders present; production‑ready mathematics.
 */
class IRTEngine
{
    private const INIT_THETA = 0.0;
    private const PRIOR_VAR  = 1.0;

    /** Logistic function */
    private static function logistic(float $x): float
    {
        return 1.0 / (1.0 + exp(-$x));
    }

    /**
     * Update a learner's ability given their response to an item.
     *
     * @param int   $userId
     * @param int   $itemId
     * @param bool  $correct
     * @return float New theta
     */
    public function updateAbility(int $userId, int $itemId, bool $correct): float
    {
        $item = DB::table('questions')->where('id', $itemId)->first(['discrimination', 'difficulty']);
        if (!$item) {
            throw new \RuntimeException("Item $itemId not found.");
        }
        [$a, $b] = [(float)$item->discrimination, (float)$item->difficulty];

        $theta = (float) DB::table('user_domain_stats')
            ->where('user_id', $userId)
            ->value('theta') ?? self::INIT_THETA;

        // Expected probability
        $p = self::logistic($a * ($theta - $b));
        // Newton–Raphson update
        $theta_new = $theta + ( ($correct ? 1.0 : 0.0) - $p ) / ($a * $p * (1 - $p) + 1e-6);

        DB::table('user_domain_stats')
            ->updateOrInsert(
                ['user_id' => $userId],
                ['theta' => $theta_new, 'updated_at' => now()]
            );

        return $theta_new;
    }

    /**
     * Select next item near learner ability.
     *
     * @param float $theta
     * @param string $domain
     * @return object Item row
     */
    public function nextItem(float $theta, string $domain)
    {
        return DB::table('questions')
            ->where('domain', $domain)
            ->orderByRaw('ABS(difficulty - ?) ASC', [$theta])
            ->first();
    }
}
