<?php
/**
 * SocialPosterService
 *
 * This service publishes content to multiple social‑media platforms
 * (Twitter / X, Facebook, LinkedIn) using their respective REST APIs.
 * API tokens are read from environment variables so that credentials
 * are never hard‑coded.
 *
 * Usage:
 *   SocialPosterService::postToSocialMedia('twitter', 'CILS B1 launches today!');
 */
class SocialPosterService
{
    public static function postToSocialMedia(string $platform, string $content): bool
    {
        switch (strtolower($platform)) {
            case 'twitter':
            case 'x':
                return self::postToTwitter($content);
            case 'facebook':
                return self::postToFacebook($content);
            case 'linkedin':
                return self::postToLinkedIn($content);
            default:
                error_log("[SocialPoster] Unsupported platform: {$platform}");
                return false;
        }
    }

    private static function postToTwitter(string $content): bool
    {
        $token = getenv('TWITTER_BEARER_TOKEN');
        $url   = 'https://api.twitter.com/2/tweets';

        $payload = json_encode(['text' => $content]);
        $ch      = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_POST           => true,
            CURLOPT_HTTPHEADER     => [
                'Content-Type: application/json',
                "Authorization: Bearer {$token}"
            ],
            CURLOPT_POSTFIELDS     => $payload,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT        => 10,
        ]);
        $response = curl_exec($ch);
        $status   = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        return $status === 201;
    }

    private static function postToFacebook(string $content): bool
    {
        $pageId  = getenv('FB_PAGE_ID');
        $token   = getenv('FB_ACCESS_TOKEN');
        $url     = "https://graph.facebook.com/{$pageId}/feed";

        $payload = http_build_query([
            'message'      => $content,
            'access_token' => $token,
        ]);
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => $payload,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT        => 10,
        ]);
        $response = curl_exec($ch);
        $status   = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        return $status === 200;
    }

    private static function postToLinkedIn(string $content): bool
    {
        $token        = getenv('LINKEDIN_ACCESS_TOKEN');
        $organization = getenv('LINKEDIN_ORG_ID');
        $url          = 'https://api.linkedin.com/v2/ugcPosts';

        $payload = json_encode([
            "author"          => "urn:li:organization:{$organization}",
            "lifecycleState"  => "PUBLISHED",
            "specificContent" => [
                "com.linkedin.ugc.ShareContent" => [
                    "shareCommentary" => ["text" => $content],
                    "shareMediaCategory" => "NONE"
                ]
            ],
            "visibility" => ["com.linkedin.ugc.MemberNetworkVisibility" => "PUBLIC"]
        ], JSON_UNESCAPED_SLASHES);

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_POST           => true,
            CURLOPT_HTTPHEADER     => [
                'Content-Type: application/json',
                "Authorization: Bearer {$token}"
            ],
            CURLOPT_POSTFIELDS     => $payload,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT        => 10,
        ]);
        $response = curl_exec($ch);
        $status   = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        return in_array($status, [201, 202], true);
    }
}
?>